/*
 * Enemy Territory Minimizer Pro 0.24
 *
 * Copyright  2003 Matthew Vilcsak
 * Copyright  2007 nano <nano@fooo.org> http://nano.fooo.org/
 * Copyright  2008-09 Anthales <rothdavid@web.de> http://anthales.de.vu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <windows.h>
#include <shellapi.h>
#include <stdio.h>  //fopen etc
#include <stdlib.h> //malloc, free
#include <cstring>  //strcpy, strlen
#include <cmath>    //pow
#include <vector>

#if !defined(max)
#define max(a, b) ((a) > (b) ? (a) : (b))
#define min(a, b) ((a) < (b) ? (a) : (b))
#endif

using std::vector;

#include "resource.h"

#define TRAYICON_ID 402442
#define TRAYICON_CALLBACK WM_USER + 14
#define HOTKEY_ALT_X 40023
#define HOTKEY_CTRL_ALT_M 40024


vector <const char*> gameList;
unsigned int internGames = 0;
//DEVMODE winRes, gameRes;
WORD latest_gamma[3][256];
HMENU hmenu;
HINSTANCE hInstance;

typedef BOOL (WINAPI *SetDeviceGammaRamp_t)(HDC hDC, LPVOID lpRamp);

SetDeviceGammaRamp_t pGetDeviceGammaRamp;
SetDeviceGammaRamp_t pSetDeviceGammaRamp;

/* --- */

void InitGameList() {
	gameList.push_back("Enemy Territory");
	gameList.push_back("Wolfenstein");
	gameList.push_back("Quake 3: Arena");
	gameList.push_back("Quake3");
	gameList.push_back("CNQ3");
	gameList.push_back("OpenArena");
	gameList.push_back("icculus.||g/quake3");
	gameList.push_back("ETQW");
	gameList.push_back("WolfMP");
	gameList.push_back("Valve001");
	gameList.push_back("DoomsdayMainWClass");
	gameList.push_back("Jedi Knight\x3F: Jedi Academy (MP)");
	gameList.push_back("SoF2 MP");
	gameList.push_back("SDL_app");
	internGames = gameList.size();

	FILE *pFile;
	pFile = fopen ("gamelist.txt", "r");
	if (pFile == NULL) return;

	int i, c;
	char *buffer;
	do {
		buffer = (char *)malloc(64 * sizeof(char));
		if (buffer == NULL) {
			MessageBox(NULL, "Couldn't allocate memory for games from gamelist.txt.\nSome custom games may not minimize.", NULL, MB_DEFBUTTON1);
			break;
		}

		//Read in line of up to 63 chars
		for (i = 0; i < 63; i++) {
			c = fgetc(pFile);
			if (c == EOF || c == '\x0D' || c == '\x0A') {
				buffer[i] = '\x00';
				break;
			}
			buffer[i] = (char)c;
		}
		buffer[63] = '\x00';

		if (strlen(buffer)) gameList.push_back((const char*)buffer);
		else                free(buffer);
	} while (c != EOF);

	if (ferror(pFile))
		MessageBox(NULL, "Error reading gamelist.txt.\nSome custom games may not minimize.", NULL, MB_DEFBUTTON1);

	fclose(pFile);
}

void SetGamma(float gamma) {
	WORD ramp[256 * 3];
	unsigned int i;
	for (i = 0; i < 256; i++) {
		ramp[i] = ramp[i + 256] = ramp[i + 512] =
			(WORD)min(65535, max(0, pow((i + 1) / 256.0f, gamma) * 65535 + 0.5));
	}

	pSetDeviceGammaRamp(GetDC(NULL), ramp);
}

void ToggleWindow(void) {
	HWND hwnd;
	WINDOWPLACEMENT wp;

	unsigned int i, size = gameList.size();
	for (i = 0; i < size; i++) {

		if (hwnd = FindWindow(gameList.at(i), NULL)) {
			GetWindowPlacement(hwnd, &wp);

			if (wp.showCmd == SW_SHOWMINIMIZED) {
//				if (ChangeDisplaySettings(&gameRes, CDS_UPDATEREGISTRY))
//					MessageBox(NULL, "Could not change screen resolution", NULL, MB_DEFBUTTON1);

				pSetDeviceGammaRamp(GetDC(NULL), (LPVOID) latest_gamma);
				ShowWindow(hwnd, SW_RESTORE);
			} else {
//				if (!EnumDisplaySettings(NULL, ENUM_CURRENT_SETTINGS, &gameRes))
//					MessageBox(NULL, "Could not enum game screen resolution", NULL, MB_DEFBUTTON1);
//				if (ChangeDisplaySettings(&winRes, CDS_UPDATEREGISTRY))
//					MessageBox(NULL, "Could not change screen resolution", NULL, MB_DEFBUTTON1);

				pGetDeviceGammaRamp(GetDC(NULL), (LPVOID) latest_gamma);
				SetGamma(1.0);
				ShowWindow(hwnd, SW_MINIMIZE);
			}
		}
	}
}


BOOL CALLBACK AboutDialogProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam) {
	switch (msg) {
		case WM_INITDIALOG:
			return true;

		case WM_CLOSE:
			EndDialog(hwnd, 0);
			return true;

		case WM_LBUTTONDOWN:
			{
			POINT pt = {LOWORD(lParam), HIWORD(lParam)};
			HWND hwndHover = ChildWindowFromPoint(hwnd, pt);

			if (GetWindowLong(hwndHover, GWL_ID) == IDC_WEBSITE)
				ShellExecute(hwnd, "open", "http://anthales.de.vu/", NULL, NULL, SW_SHOWNORMAL);

			}
			break;

		case WM_CTLCOLORSTATIC:
			{
			DWORD dwId = GetWindowLong((HWND)lParam, GWL_ID);
			HDC hdc = (HDC)wParam;

			if (dwId == IDC_WEBSITE) {
				SetBkMode(hdc, TRANSPARENT);

				if (GetSysColorBrush(26)) SetTextColor(hdc, GetSysColor(26));
				else                      SetTextColor(hdc, RGB(0, 0, 255));

				return (LONG)GetSysColorBrush(COLOR_BTNFACE);
			}
			}
			break;

		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDOK:
				case IDCANCEL:
					EndDialog(hwnd, IDOK);
					return true;
				default:
					break;
			}
			return false;
	}

	return false;
}

LRESULT CALLBACK WinProc(HWND hwnd, UINT msg, WPARAM wp, LPARAM lp) {
	switch (msg) {
		case WM_HOTKEY:
			if (wp == HOTKEY_ALT_X || wp == HOTKEY_CTRL_ALT_M)
				ToggleWindow();
			break;

		case TRAYICON_CALLBACK:
			if (lp == WM_RBUTTONDOWN) {
				POINT p;

				SetForegroundWindow(hwnd);
				GetCursorPos(&p);
				TrackPopupMenu(hmenu, 0, p.x, p.y, 0, hwnd, 0);
				PostMessage(hwnd, WM_NULL, 0,0);
			} else if (lp == WM_LBUTTONDBLCLK) {
				ToggleWindow();
			}
			break;

		case WM_COMMAND:
			switch (LOWORD(wp)) {
				case IDM_TOGGLE:
					ToggleWindow();
					break;
				case IDM_EXIT:
					DestroyWindow(hwnd);
					break;
				case IDM_GAMMA:
					SetGamma(1.0);
					break;
				case IDM_ABOUT:
					DialogBox(hInstance, MAKEINTRESOURCE(IDD_ABOUT), hwnd, AboutDialogProc);
					break;
			}
			break;

		case WM_DESTROY:
			PostQuitMessage(0);
			break;

		default:
			return DefWindowProc(hwnd, msg, wp, lp);
	}

	return true;
}

int WINAPI WinMain(HINSTANCE hInst, HINSTANCE hPrev, LPSTR lpCmdLine, int nCmdShow) {
	HWND hwnd;
	MSG msg;
	WNDCLASSEX wc;
	HMODULE hGDI32;
	NOTIFYICONDATA trayicon;

	hInstance = hInst;

	wc.cbSize = sizeof(WNDCLASSEX);
	wc.cbClsExtra = 0;
	wc.cbWndExtra = 0;
	wc.hbrBackground = (HBRUSH) WHITE_BRUSH + 1;
	wc.hCursor = NULL;
	wc.hIcon = LoadIcon(hInst, MAKEINTRESOURCE(ETMINPROICON));
	wc.hIconSm = NULL;
	wc.hInstance = hInst;
	wc.lpfnWndProc = WinProc;
	wc.lpszClassName = "etminpro";
	wc.style = 0;
	wc.lpszMenuName = NULL;

	if (!RegisterClassEx(&wc))
		return 1;

	hwnd = CreateWindowEx(WS_EX_CLIENTEDGE, "etminpro", "etminpro",
		WS_OVERLAPPEDWINDOW, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
		CW_USEDEFAULT, NULL, NULL, hInst, NULL);

	ZeroMemory(&trayicon, sizeof(NOTIFYICONDATA));
	strcpy(trayicon.szTip, "Enemy Territory Minimizer Pro");

	trayicon.cbSize = sizeof(NOTIFYICONDATA);
	trayicon.hWnd = hwnd;
	trayicon.uID = TRAYICON_ID;
	trayicon.uFlags = NIF_ICON | NIF_TIP | NIF_MESSAGE;
	trayicon.uCallbackMessage = TRAYICON_CALLBACK;
	trayicon.hIcon = LoadIcon(hInst, MAKEINTRESOURCE(ETMINPROICON));

	Shell_NotifyIcon(NIM_ADD, &trayicon);
	hmenu = GetSubMenu(LoadMenu(hInst, MAKEINTRESOURCE(IDM_MENU)), 0);

	if (!RegisterHotKey(NULL, HOTKEY_ALT_X, MOD_ALT, 'X') ||
			!RegisterHotKey(NULL, HOTKEY_CTRL_ALT_M, MOD_ALT | MOD_CONTROL, 'M')) {
		MessageBox(NULL, "Could not register hotkey.", NULL, MB_DEFBUTTON1);
		PostQuitMessage(1);
	}

	if ((hGDI32 = LoadLibrary("gdi32.dll")) != NULL) {
		pGetDeviceGammaRamp = (SetDeviceGammaRamp_t) GetProcAddress(hGDI32, "GetDeviceGammaRamp");
		pSetDeviceGammaRamp = (SetDeviceGammaRamp_t) GetProcAddress(hGDI32, "SetDeviceGammaRamp");
	} else {
		MessageBox(NULL, "Could not load library gdi32.dll", NULL, MB_DEFBUTTON1);
		PostQuitMessage(2);
	}

//	if (!EnumDisplaySettings(NULL, ENUM_CURRENT_SETTINGS, &winRes))
//		MessageBox(NULL, "Could not enum screen resolution", NULL, MB_DEFBUTTON1);

	InitGameList();

	while (GetMessage(&msg, NULL, 0, 0)) {
		if (msg.message == WM_HOTKEY)
			msg.hwnd = hwnd;

		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

	UnregisterHotKey(NULL, HOTKEY_ALT_X);
	UnregisterHotKey(NULL, HOTKEY_CTRL_ALT_M);

	Shell_NotifyIcon(NIM_DELETE, &trayicon);

	unsigned int i, size = gameList.size();
	for (i = internGames; i < size; i++)
		free((char *)gameList.at(i));
	gameList.clear();

	return msg.wParam;
}
